'
'  dd_link.bas
'
'  DESCRIPTION
'     Linked list handling. This is an alternative to
'     using arrays.
'
'  AUTHOR
'     Don Dickinson
'     ddickinson@usinternet.com
'     http://dickinson.basicguru.com
'
'  LICENSE and DISCLAIMER
'     Hereby Public Domain
'
'     Use this code as you see fit. By using or compiling this code or derivative
'     thereof, you are consenting to the hold the author, Don Dickinson, harmless
'     for all effects or side-effects its use. This code works great for me,
'     but you are using it at your own risk.
'
'  DEPENDS ON
'     pb_mem.bas     memory handling functions from Don Dickinson
'     win32api.inc   pb's windows api declarations
'
#if not %def(%DD_LINK_BAS)

%DD_LINK_BAS = 1

#if not %def(%WINAPI)
#include "win32api.inc"
#endif
#include "pb_mem.bas"

Type LINKED_LIST_TYPE
   pData as Asciiz Ptr
   pName as Asciiz Ptr
   dataCapacity as Long
   pNext as LINKED_LIST_TYPE PTR
End Type

$DD_LINKED_LIST_FILE_ID = "LINKED_LIST_TYPE"
%LL_SORT_STRING   = 0
%LL_SORT_NUMBER   = 1

Global g_llz_Return as String
Global g_ll_LastError as String

'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
'  llFindLast
'     Returns a pointer to the last node in the list
'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
Function llFindLast Alias "llFindLast" _
      ( pFirst as LINKED_LIST_TYPE PTR ) Export as DWord

   Dim pLast as LINKED_LIST_TYPE PTR

   if pFirst then
      pLast = pFirst
      do until @pLast.pNext = 0
         pLast = @pLast.pNext
      loop
      Function = pLast
   else
      Function = 0
   end if

End Function

'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
'  llCount
'     Returns the number of nodes in the linked list
'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
Function llCount Alias "llCount" _
      ( pFirst as LINKED_LIST_TYPE PTR ) Export as Long

   Dim iCount as Long
   Dim pNext as LINKED_LIST_TYPE PTR

   iCount = 0

   pNext = pFirst
   Do until pNext = 0
      incr iCount
      pNext = @pNext.pNext
   Loop
   Function = iCount

End Function

'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
'  llGetByNumber
'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
Function llGetByNumber Alias "llGetByNumber" _
      ( pFirst as LINKED_LIST_TYPE PTR, ByVal iWhich as Long ) export as DWord

   Dim iCount as Long
   Dim pNext as LINKED_LIST_TYPE PTR
   Dim pFound as DWord

   iCount = 0
   pFound = 0
   pNext = pFirst
   Do until pNext = 0
      incr iCount
      if iCount = iWhich then
         pFound = pNext
         exit do
      end if
      pNext = @pNext.pNext
   Loop
   Function = pFound

End Function

'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
'  llAdd
'     Adds an item to the end of a linked list. Returns a pointer to
'     that item.
'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
Function llAdd Alias "llAdd" _
      ( pNode as LINKED_LIST_TYPE PTR, ByVal sName as String, _
        ByVal sData as String) Export as Dword

   Dim pNext as LINKED_LIST_TYPE PTR
   Dim pLast as LINKED_LIST_TYPE PTR

   pNext = GetMem(len(@pNext))
   if pNext = 0 then
      Function = 0
      exit function
   end if

   @pNext.pData = GetMem(len(sData) + 1)
   if @pNext.pData = 0 then
      FreeMem pNext
      Function = 0
      exit function
   end if

   @pNext.@pData = sData
   @pNext.dataCapacity = len(sData)
   sName = trim$(lcase$(sName))
   @pNext.pName = GetAsciiz(sName)

   '- Add this item to the end of the list
   pLast = llFindLast(ByVal pNode)
   if pLast then
      @pLast.pNext = pNext
   end if

   Function = pNext

End Function

'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
'  llzAdd
'     Adds an item to the end of a linked list. Returns a pointer to
'     that item.
'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
Function llzAdd Alias "llzAdd" _
      ( pNode as LINKED_LIST_TYPE PTR, zName as Asciiz, _
        zData as Asciiz) Export as Dword

   Function = llAdd(pNode, (zName), (zData))

End Function

'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
'  llbAdd
'     Adds binary data to the end of a linked list. Returns a pointer to
'     that item.
'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
Function llbAdd Alias "llbAdd" _
      ( pNode as LINKED_LIST_TYPE PTR, ByVal sName as String, _
        ByVal sData as String) Export as Dword

   Dim pNext as LINKED_LIST_TYPE PTR
   Dim pLast as LINKED_LIST_TYPE PTR

   pNext = GetMem(len(@pNext))
   if pNext = 0 then
      Function = 0
      exit function
   end if

   @pNext.pData = GetMem(len(sData))
   if @pNext.pData = 0 then
      FreeMem pNext
      Function = 0
      exit function
   end if

   CopyMemory @pNext.pData, ByVal strptr(sData), len(sData)
   @pNext.dataCapacity = len(sData)
   sName = trim$(lcase$(sName))
   @pNext.pName = GetAsciiz(sName)

   '- Add this item to the end of the list
   pLast = llFindLast(ByVal pNode)
   if pLast then
      @pLast.pNext = pNext
   end if

   Function = pNext

End Function

'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
'  llFree
'     Frees a list
'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
Sub llFree Alias "llFree" ( pFirst as LINKED_LIST_TYPE PTR ) Export

   Dim pNext as LINKED_LIST_TYPE PTR
   Dim pSave as Dword

   pNext = pFirst
   Do until pNext = 0
      if @pNext.pData then
         FreeMem @pNext.pData
      end if
      if @pNext.pName then
         FreeMem @pNext.pName
      end if
      pSave = pNext
      pNext = @pNext.pNext
      FreeMem pSave
   Loop

End Sub

'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
'  llUpdate
'     Updates an item in the list.
'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
Sub llUpdate Alias "llUpdate" _
      ( pItem as LINKED_LIST_TYPE PTR, ByVal sItem as String ) Export

   if pItem = 0 then
      exit sub
   end if

   if @pItem.pData then
      FreeMem @pItem.pData
      @pItem.pData = 0
      @pItem.dataCapacity = 0
   end if
   @pItem.pData = GetMem(len(sItem) + 1)
   if @pItem.pData then
      @pItem.@pData = sItem
      @pItem.dataCapacity = len(sItem)
   end if

End Sub

'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
'  llzUpdate
'     Updates an item in the list.
'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
Sub llzUpdate Alias "llzUpdate" _
      ( pItem as LINKED_LIST_TYPE PTR, zItem as Asciiz) Export

   llUpdate pItem, (zItem)

End Sub

'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
'  llbUpdate
'     Updates an item with binary data in the list.
'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
Sub llbUpdate Alias "llbUpdate" _
      ( pItem as LINKED_LIST_TYPE PTR, ByVal sItem as String ) Export

   if pItem = 0 then
      exit sub
   end if

   if @pItem.pData then
      FreeMem @pItem.pData
      @pItem.pData = 0
      @pItem.dataCapacity = 0
   end if

   @pItem.pData = GetMem(len(sItem))
   if @pItem.pData then
      CopyMemory @pItem.pData, byval strptr(sItem), len(sItem)
      @pItem.dataCapacity = len(sItem)
   end if

End Sub

'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
'  llFindByName
'     Locates a node by its name. Returns a pointer to the found node
'     or Zero if it can't be found.
'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
Function llFindByName Alias "llFindByName" _
      ( pFirst as LINKED_LIST_TYPE PTR, ByVal sName as String ) _
        Export as DWord

   Dim pNext as LINKED_LIST_TYPE PTR

   Function = 0
   sName = trim$(lcase$(sName))
   pNext = pFirst
   Do until pNext = 0
      if @pNext.pName then
         if @pNext.@pName = sName then
            Function = pNext
            exit do
         end if
      end if
      pNext = @pNext.pNext
   Loop

End Function

'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
'  llzFindByName
'     Locates a node by its name. Returns a pointer to the found node
'     or Zero if it can't be found.
'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
Function llzFindByName Alias "llzFindByName" _
      ( pFirst as LINKED_LIST_TYPE PTR, zName as Asciiz ) _
        Export as DWord

   Function = llFindByname(pFirst, (zName) )

End Function

'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
'  llFindByData
'     Locates a node by its data. Returns a pointer to the found node
'     or Zero if it can't be found.
'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
Function llFindByData Alias "llFindByData" _
      ( pFirst as LINKED_LIST_TYPE PTR, ByVal sData as String ) _
        Export as DWord

   Dim pNext as LINKED_LIST_TYPE PTR

   Function = 0
   sData = trim$(lcase$(sData))
   pNext = pFirst
   Do until pNext = 0
      if zToString(@pNext.pData) = sData then
         Function = pNext
         exit do
      end if
      pNext = @pNext.pNext
   Loop

End Function

'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
'  llzFindByData
'     Locates a node by its data. Returns a pointer to the found node
'     or Zero if it can't be found.
'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
Function llzFindByData Alias "llzFindByData" _
      ( pFirst as LINKED_LIST_TYPE PTR, zData as Asciiz ) _
        Export as DWord

   Function = llFindByData(pFirst, (zData) )

End Function

'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
'  llUpdateByName
'     Updates the specified item. If it can't be found, then
'     it creates a new item. The pointer to the item is returned.
'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
Function llUpdateByName Alias "llUpdateByName" _
      ( pFirst as LINKED_LIST_TYPE PTR, ByVal sName as String, _
        ByVal sItem as String ) Export as DWord

   Dim pFind as LINKED_LIST_TYPE PTR

   pFind = llFindByName(ByVal pFirst, sName)
   if pFind = 0 then
      pFind = llAdd(ByVal pFirst, sName, sItem)
   else
      llUpdate ByVal pFind, sItem
   end if

   Function = pFind

End Function

'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
'  llbUpdateByName
'     Updates the specified item that holds binary data.
'     If it can't be found, then
'     it creates a new item. The pointer to the item is returned.
'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
Function llbUpdateByName Alias "llbUpdateByName" _
      ( pFirst as LINKED_LIST_TYPE PTR, ByVal sName as String, _
        ByVal sItem as String ) Export as DWord

   Dim pFind as LINKED_LIST_TYPE PTR

   pFind = llFindByName(ByVal pFirst, sName)
   if pFind = 0 then
      pFind = llbAdd(ByVal pFirst, sName, sItem)
   else
      llbUpdate ByVal pFind, sItem
   end if

   Function = pFind

End Function

'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
'  llzUpdateByName
'     Updates the specified item. If it can't be found, then
'     it creates a new item. The pointer to the item is returned.
'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
Function llzUpdateByName Alias "llzUpdateByName" _
      ( pFirst as LINKED_LIST_TYPE PTR, zName as Asciiz, _
        zItem as Asciiz) Export as DWord

   Function = llUpdateByName(pFirst, (zName), (zItem) )

End Function

'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
'  llData
'     Returns the data stored in a linked list node
'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
Function llData Alias "llData" _
      ( pItem as LINKED_LIST_TYPE PTR ) Export as String

   if pItem = 0 then
      Function = ""
   elseif @pItem.pData = 0 then
      Function = ""
   else
      Function = @pItem.@pData
   end if

End Function

'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
'  llzData
'     Returns the data stored in a linked list node
'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
Function llzData Alias "llzData" _
      ( pItem as LINKED_LIST_TYPE PTR ) Export as DWord

   g_llz_Return = llData(pItem) + $nul
   Function = StrPtr(g_llz_Return)

End Function

'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
'  llbData
'     Returns the binary data stored in a linked list node
'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
Function llbData Alias "llbData" _
      ( pItem as LINKED_LIST_TYPE PTR ) Export as String

   Dim sReturn as String
   if pItem = 0 then
      Function = ""
   elseif @pItem.pData = 0 then
      Function = ""
   else
      sReturn = Space$(@pItem.dataCapacity)
      CopyMemory ByVal StrPtr(sReturn), @pItem.pData, len(sReturn)
      Function = sReturn
   end if

End Function

'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
'  llName
'     Returns the name stored in a linked list node
'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
Function llName Alias "llName" _
      ( pItem as LINKED_LIST_TYPE PTR ) Export as String

   if pItem = 0 then
      Function = ""
   elseif @pItem.pName = 0 then
      Function = ""
   else
      Function = @pItem.@pName
   end if

End Function

'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
'  llzName
'     Returns the name stored in a linked list node
'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
Function llzName Alias "llzName" _
      ( pItem as LINKED_LIST_TYPE PTR ) Export as DWord

   g_llz_Return = llName(pItem) + $nul
   function = strptr(g_llz_Return)

End Function

'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
'  llDataByName
'     Returns the list item's data given its name
'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
Function llDataByName Alias "llDataByName" _
      ( pFirst as LINKED_LIST_TYPE PTR, ByVal sName as String ) _
        Export as String

   Dim pFind as LINKED_LIST_TYPE PTR

   pFind = llFindByName( ByVal pFirst, sName )
   Function = llData(ByVal pFind)

End Function

'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
'  llzDataByName
'     Returns the list item's data given its name
'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
Function llzDataByName Alias "llzDataByName" _
      ( pFirst as LINKED_LIST_TYPE PTR, zName as Asciiz ) _
        Export as Dword

   g_llz_Return = llDataByName(pFirst, (zName) ) + $nul
   function = strptr(g_llz_Return)

End Function

'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
'  llbDataByName
'     Returns the list item's binary data given its name
'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
Function llbDataByName Alias "llbDataByName" _
      ( pFirst as LINKED_LIST_TYPE PTR, ByVal sName as String ) _
        Export as String

   Dim pFind as LINKED_LIST_TYPE PTR

   pFind = llFindByName( ByVal pFirst, sName )
   Function = llbData(ByVal pFind)

End Function

'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
'  llNext
'     Returns a pointer to the next node
'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
Function llNext Alias "llNext" _
      ( pItem as LINKED_LIST_TYPE PTR ) Export as DWord

   if pItem = 0 then
      function = 0
   else
      function = @pItem.pNext
   end if

End Function

'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
'  llFindPrev
'     Returns a pointer to the list item before the one specified.
'     Returns 0 if not found or if pFind is pFirst
'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
Function llFindPrev Alias "llFindPrev" _
      ( pFirst as LINKED_LIST_TYPE PTR, pFind as LINKED_LIST_TYPE PTR ) _
        Export as DWord

   Dim pPrev as LINKED_LIST_TYPE PTR
   Dim pCheck as LINKED_LIST_TYPE PTR

   '- Set the default
   Function = 0

   if pFirst = pFind then
      exit function
   end if

   pPrev = pFirst
   pCheck = @pFirst.pNext
   do
      '- Found it
      if pCheck = pFind then
         Function = pPrev
         exit do

      '- Not found
      elseif pCheck = 0 then
         exit do

      end if

      pPrev = pCheck
      pCheck = @pCheck.pNext
   Loop


End Function

'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
'  llDelete
'     Removes a node from the list and frees its memory
'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
Function llDelete Alias "llDelete" _
      ( pFirst as LINKED_LIST_TYPE PTR, pDelete as LINKED_LIST_TYPE PTR) _
        Export as Long

   Dim pPrev as LINKED_LIST_TYPE PTR

   if pDelete = 0 then
      Function = %false
      exit function
   end if

   '- Find the node before the one to be deleted
   pPrev = llFindPrev(ByVal pFirst, ByVal pDelete)

   '- Remove pDelete from the list
   if pPrev then
      @pPrev.pNext = @pDelete.pNext
   end if

   '- Free pDelete (and only pDelete)
   @pDelete.pNext = 0
   llFree ByVal pDelete

   Function = %True

End Function

'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
'  llDeleteLikeNodes
'     Deletes all nodes whose name starts with sLike.
'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
Sub llDeleteLikeNodes Alias "llDeleteLikeNodes" _
      ( pFirst as LINKED_LIST_TYPE PTR, ByVal sLike as String )

   Dim sThis as String
   Dim pNext as LINKED_LIST_TYPE PTR
   Dim pThis as LINKED_LIST_TYPE PTR

   sLike = trim$(lcase$(sLike))
   if sLike = "" then exit sub

   pThis = pFirst
   Do until pThis = 0
      pNext = @pThis.pNext
      sThis = trim$(lcase$(llName(pThis)))
      if sThis <> "" then
         if left$(sThis, len(sLike)) = sLike then
            llDelete ByVal pFirst, ByVal pThis
         end if
      end if
      pThis = pNext
   Loop

End Sub

'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
'  llzDeleteLikeNodes
'     Deletes all nodes whose name starts with sLike.
'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
Sub llzDeleteLikeNodes Alias "llzDeleteLikeNodes" _
      ( pFirst as LINKED_LIST_TYPE PTR, zLike as Asciiz )

   llDeleteLikeNodes pFirst, (zLike)

End Sub

'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
'  llDeleteByName
'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
Function llDeleteByName Alias "llDeleteByName" _
      ( pFirst as LINKED_LIST_TYPE PTR, ByVal sName as String ) _
        Export as Long

   Dim pFind as LINKED_LIST_TYPE PTR

   pFind = llFindByname(ByVal pFirst, sName)
   if pFind then
      Function = llDelete(ByVal pFirst, ByVal pFind)
   else
      Function = %true
   end if

End Function

'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
'  llzDeleteByName
'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
Function llzDeleteByName Alias "llzDeleteByName" _
      ( pFirst as LINKED_LIST_TYPE PTR, zName as Asciiz ) Export as Long

   Function = llzDeleteByName(pFirst, (zName) )

End Function

'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
'  llCapacity
'     Returns the capacity of the specified node
'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
Function llCapacity Alias "llCapacity" _
      ( pItem as LINKED_LIST_TYPE PTR ) Export as Long

   if pItem = 0 then
      Function = 0
   else
      Function = @pItem.dataCapacity
   end if

End Function

'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
'  llCapacityByName
'     Returns the node's capacity give the node name
'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
Function llCapacityByName Alias "llCapacityByName" _
      ( pFirst as LINKED_LIST_TYPE PTR, ByVal sName as String ) as Long

   Dim pFind as LINKED_LIST_TYPE PTR

   pFind = llFindByName(ByVal pFirst, sName )
   Function = llCapacity(ByVal pFind)

End Function

'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
'  llzCapacityByName
'     Returns the node's capacity give the node name
'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
Function llzCapacityByName Alias "llzCapacityByName" _
      ( pFirst as LINKED_LIST_TYPE PTR, zName as Asciiz ) as Long


   Function = llCapacityByName(pFirst, (zName) )

End Function

'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
'  llMoveFirst
'     Moves a node to first in the list.
'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
Sub llMoveFirst Alias "llMoveFirst" _
      ( pFirst as LINKED_LIST_TYPE PTR, pNewFirst as LINKED_LIST_TYPE PTR, _
        pBeforeNew as LINKED_LIST_TYPE PTR ) Export

   if pNewFirst = 0 then exit sub

   '- Remove the new first from the line
   if pBeforeNew then
      @pBeforeNew.pNext = @pNewFirst.pNext
   end if

   '- Move it to the beginning
   @pNewFirst.pNext = pFirst

End Sub

'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
'  llFindSmallest
'     Returns a pointer to the node that contains the smallest data value.
'     It can find by number or by text.
'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
Function llFindSmallest Alias "llFindSmallest" _
      ( pFirst as LINKED_LIST_TYPE PTR, ByVal iSortType as Long, _
        pPrev as Dword ) Export as Dword

   Dim pNext as LINKED_LIST_TYPE PTR
   Dim pSmallest as LINKED_LIST_TYPE PTR
   Dim pLastPrev as LINKED_LIST_TYPE PTR

   pPrev = 0
   pNext = pFirst
   do until pNext = 0
      if @pNext.pData then
         if pSmallest = 0 then
            pSmallest = pNext
            pPrev = pLastPrev
         elseif iSortType = %LL_SORT_NUMBER then
            if val(@pNext.@pData) < val(@pSmallest.@pData) then
               pSmallest = pNext
               pPrev = pLastPrev
            end if
         elseif @pNext.@pData < @pSmallest.@pData then
            pSmallest = pNext
            pPrev = pLastPrev
         end if
      end if
      pLastPrev = pNext
      pNext = @pNext.pNext
   loop

   Function = pSmallest

End Function

'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
'  llSort
'     Sorts the linked list. Returns a pointer to the new first list item.
'     This function is to be considered beta. it can be slow and isn't well
'     tested.
'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
Function llSort Alias "llSort" _
      ( pFirst as LINKED_LIST_TYPE PTR, ByVal iSortType as Long ) _
        Export as DWord

   Dim pNext as LINKED_LIST_TYPE PTR
   Dim pSmallest as LINKED_LIST_TYPE PTR
   Dim pSorted as LINKED_LIST_TYPE PTR
   Dim pPrev as LINKED_LIST_TYPE PTR
   Dim pLast as LINKED_LIST_TYPE PTR

   pSorted = 0
   pNext = pFirst
   pSmallest = llFindSmallest(ByVal pNext, iSortType, ByVal pPrev)
   Do until pSmallest = 0

      '- If there was a previous item, then
      '  we have to point it to the next
      '  one in the list.
      '
      if pPrev then
         @pPrev.pNext = @pSmallest.pNext

      '- Here is the new root node since we
      '  removed the last one.
      '
      else
         pNext = @pSmallest.pNext

      end if

      '- If we don't have a root for the sorted
      '  items, then this is it.
      '
      if pSorted = 0 then
         pSorted = pSmallest
      end if

      @pSmallest.pNext = 0

      '- Move the smallest over to
      '  the end of the sorted list.
      '
      pLast = llFindLast(ByVal pSorted)
      if pLast <> pSmallest then
         @pLast.pNext = pSmallest
      end if

      pSmallest = llFindSmallest(ByVal pNext, iSortType, ByVal pPrev)
   Loop

   if pSorted = 0 then
      pSorted = pFirst
   end if
   Function = pSorted

End Function

'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
'  llLastError
'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
Function llLastError Alias "llLastError" Export as String

   Function = g_ll_LastError

End Function

'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
'  llzLastError
'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
Function llzLastError Alias "llzLastError" Export as Dword

   g_llz_Return = g_ll_LastError + $nul
   Function = strptr(g_llz_Return)

End Function

'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
'  llListFromFile
'     Returns a pointer to the linked list contained in the data file.
'     Returns 0 and sets llLastError if there was an errr.
'     The programmer is responsible for calling llFree to
'     free memory allocated for the pointer returned.
'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
Function llListFromFile Alias "llListFromFile" _
      ( ByVal sFileName as String ) _
      Export as DWord

   Dim iFF as Long
   Dim iOk as Long
   Dim iNameLen as Long
   Dim iDataLen as Long
   Dim sName as String
   Dim sData as String
   Dim pReturn as LINKED_LIST_TYPE PTR

   iOk = %true

   '- Make sure the file exists.
   if dir$(sFileName) = "" then
      g_ll_LastError = "Input file '" + sFileName + "' doesn't exist."
      iOk = %false
      goto llListFromFile_Done
   end if

   '- Open the file
   iFF = FreeFile
   Open sFileName for binary lock read write as #iFF
   if err then
      iOk = %false
      g_ll_LastError = "Unable to open file " + sFileName + _
            " Error = " + Format$(Err)
      iFF = 0
      Goto llListFromFile_Done
   end if

   '- Check for the linked list signature.
   sName = Space$(len($DD_LINKED_LIST_FILE_ID))
   Get #iFF,, sName
   if sName <> $DD_LINKED_LIST_FILE_ID then
      iOk = %false
      g_ll_LastError = "File " + sFileName + " doesn't contain " + _
            "a properly formatted linked list."
      Goto llListFromFile_Done
   end if

   errclear

   '- Read through the file
   Do
      sName = ""
      sData = ""
      iNameLen = -1
      iDataLen = -1

      Get #iFF,, iNameLen

      '- End of file marker
      if iNameLen < 0 then
         exit do
      end if

      '- Read name value
      if iNameLen > 0 then
         sName = space$(iNameLen)
         Get #iFF,, sName
      end if


      Get #iFF,, iDataLen

      '- Check for bad data - this
      '  shouldn't happen.
      '
      if iDataLen < 0 then
         exit do
      end if

      '- Read the data value
      if iDataLen > 0 then
         sData = Space$(iDataLen)
         Get #iFF,, sData
      end if

      '- Check for an I/O error
      if err then
         iOk = %false
         g_ll_LastError = "Error reading input file " + sFileName + _
               " Error = " + Format$(Err)
         Goto llListFromFile_Done
         exit do
      end if

      '- Update the list
      if pReturn = 0 then
         pReturn = llAdd(pReturn, sName, sData)
      else
         llAdd pReturn, sName, sData
      end if

   Loop

llListFromFile_Done:
   if iFF then Close #iFF
   if iOk = %false then
      if pReturn then
         llFree pReturn
         pReturn = 0
      end if
   end if

   Function = pReturn

End Function

'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
'  llListToFile
'     Saves a linked list to a disk file.
'     Returns %true on success
'     Returns %false on error. Also sets llLastError with error msg.
'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
Function llListToFile Alias "llListToFile" _
      ( ByVal pList as LINKED_LIST_TYPE PTR, ByVal sFileName as String ) _
      Export as Long

   Dim iOk as Long
   Dim iFF as Long
   Dim iNameLen as Long
   Dim iDataLen as Long
   Dim sName as String
   Dim sData as String
   Dim sOut as String
   Dim pNext as LINKED_LIST_TYPE PTR

   iOk = %true

   if dir$(sFileName) <> "" then
      g_ll_LastError = "Unable to save linked list. File " + _
            sFileName + " already exists."
      iOk = %false
      Goto llListToFile_Done
   end if

   iFF = FreeFile
   Open sFileName for binary lock read write as #iFF
   if err then
      g_ll_LastError = "Unable to create save file " + sFileName + _
            "Error = " + format$(Err)
      iFF = 0
      iOk = %false
      Goto llListToFile_Done
   end if

   '- Write out the file signature
   sName = $DD_LINKED_LIST_FILE_ID
   Put #iFF,, sName

   errclear

   '- Go through the list and write
   '  all name/value pairs out.
   '
   pNext = pList
   do until pNext = 0
      sName = llName(pNext)
      sData = llData(pNext)

      iNameLen = len(sName)
      iDatalen = len(sData)
      sOut = MkL$(iNameLen)
      Put #iFF,, sOut
      if iNameLen > 0 then
         Put #iFF,, sName
      end if

      sOut = MkL$(iDataLen)
      Put #iFF,, sOut
      if iDataLen > 0 then
         Put #iFF,, sData
      end if

      if err then
         iOk = %false
         g_ll_LastError = "i/o error writing to file " + sFileName
         goto llListToFile_Done
      end if
      pNext = @pNext.pNext
   loop

   sOut = MkL$(-1)
   Put #iFF,, sOut

llListToFile_Done:
   if iFF then Close #iFF
   Function = iOk

End Function

'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
'  llzListToFile
'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
Function llzListToFile Alias "llzListToFile" _
      ( ByVal pList as LINKED_LIST_TYPE PTR, sFileName as Asciiz ) _
      Export as Long

   Function = llListToFile( pList, (sFileName) )

End Function

'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
'  llzListFromFile
'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
Function llzListFromFile Alias "llzListFromFile" _
      ( sFileName as Asciiz ) _
      Export as DWord

   Function = llListFromFile( (sFileName) )

End Function

'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
'  ll_xmlClean
'     Removes special characters for strings so that they will
'     display correctly without messing up XML. This is used
'     when the data coming in was typed by a user.
'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
Function ll_xmlClean Alias "ll_xmlClean" _
      ( ByVal incoming as String ) Export as String

   Replace "&" with "&amp;" in incoming
   Replace "<" with "&lt;" in incoming
   Replace ">" with "&gt;" in incoming
   Replace chr$(34) with "&quot;" in incoming
   Replace "'" with "&apos;" in incoming

   Function = incoming

End Function


'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
'  ll_xmlUnClean
'     Replaces special xml character tokens with their actual
'     ascii value. This is the "un-do" for xmlClean
'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
Function ll_xmlUnClean Alias "ll_xmlUnClean" _
      ( ByVal incoming as String ) Export as String

   Replace "&apos;" with "'" in incoming
   Replace "&quot;" with chr$(34) in incoming
   Replace "&gt;" with ">" in incoming
   Replace "&lt;" with "<" in incoming
   Replace "&amp;" with "&" in incoming

   Function = incoming

End Function

'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
'  llDelimToList
'
'  Takes a delimited list (sDelim) and creates a linked list from the
'  items. Null items are ignored.
'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
function llDelimToList Alias "llDelimToList" _
      ( ByVal sList as String, ByVal sDelim as String ) Export as Dword

   Dim iPos as Long
   Dim iCount as Long
   Dim pReturn as Dword
   Dim pFirst as Dword
   Dim oneItem as String
   Dim sAccum as String
   Dim bChar as Byte ptr

   bChar = strptr(sList) - 1
   for iPos = 1 to len(sList)
      incr bChar
      if @bChar = asc(sDelim) then
         if sAccum <> "" then
            incr iCount
            if pReturn = 0 then
               pReturn = llAdd(0, format$(iCount), sAccum)
               pFirst = pReturn
            else
               pReturn = llAdd(pReturn, format$(iCount), sAccum)
            end if
            sAccum = ""
         end if
      else
         sAccum = sAccum + chr$(@bChar)
      end if
   next iPos

   if sAccum <> "" then
      incr iCount
      if pReturn = 0 then
         pReturn = llAdd(0, format$(iCount), sAccum)
         pFirst = pReturn
      else
         pReturn = llAdd(pReturn, format$(iCount), sAccum)
      end if
   end if

   function = pFirst

end function

'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
'  llzDelimToList
'
'  Takes a delimited list (sDelim) and creates a linked list from the
'  items. Null items are ignored.
'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
function llzDelimToList Alias "llzDelimToList" _
      ( zList as asciiz, zDelim as asciiz ) Export as Dword

   function = llDelimToList( (zList), (zDelim) )

end function

#endif

